% Script to create LBF representations for the ice VII-X system from 1 bar
% to 450 GPa based on methods discussed in Brown and Journaux.  The
% Helmholtz toolbox must be in the search path of the environment.  This
% script has been tested in both MATLAB and OCTAVE


% start by clearing environment and loading data
clear all ; close all
load Data_literature

%%
% select data around 300K and remove  Hemley (1987) and Loubeyre (1999)

id=find(Data.VTP.data(:,2) <= 310  &  Data.VTP.data(:,4) ~= 3 & Data.VTP.data(:,4) ~= 8);
data.PV=[Data.VTP.data(id,[3,1])];

% Sort data in order of increasing volume (necessary for underlying spline
% algorithms
data.PV = sortrows(data.PV,2);
% Define Vo
data.Vo = 12.7218 ; % in cm^3/mol Klotz et al. (2017)
data.Vflg=1;  %input in volumes

%%
%%%%%%%%%%% 4th order global fit
clear options
options.strainflg='Eulerian';
options.knt=[12.75  3.7]; %knots in volumes  NOTE: spline on one interval is equivalant to a standard polynomial 
options.k=5; %specify spline order (degree is one less)
LBF_ref=lin_F_fit(data,options);



%%
%Chose Bulk moduli Data that might be used for fits
% Compute volumes associated with the bulk modulus data by interpolation
Vc=linspace(12,6,1000);
out=fn_F_val(LBF_ref,Vc.^-1);
PZang = Data.K.Zang2019.All(:,1);
VZang=interp1(out.P,Vc,PZang,'spline'); % use volumes from global fit at pressures in Zang et al
data.K=[VZang(:) Data.K.Zang2019.All(:,5) Data.K.Zang2019.All(:,1)];

%%
%%%%%%%%%% Preferred fit
clear options
options.strainflg='log'; %Strain metric
options.Vflg=1; %inputs in volumes
options.kntflg=1; % Knots specified
%options.knt=[ -0.01 .08 0.12   0.16 0.16 0.24 0.30 0.42];
options.knt=[ -0.01 0.12 0.16 0.16 0.21 0.23 0.26 0.42];
options.Reg=[linspace(12.73,11,10) 10.85  6   5 4.5 4.2 4.1  4 3.9 3.8 3.7 ]; 
options.drv=4;  % minimization of 4th derivative of F(V) 
options.lam=1e0; % damping parameter (smoothing by minimization of 4th derivative of F(V) )
options.k=7; % Order of the spline
LBF_prf2=lin_F_fit(data,options);


clear options
options.strainflg='log'; %Strain metric
options.Vflg=1; %inputs in units of volume
options.kntflg=1; % Knots are directly specified in strain units
options.knt=[ -0.01 .08 0.12   0.16 0.16 0.24 0.30 0.42];  % the specified knots
options.Reg=[linspace(12.73,11,10) 10.85  6   5 4.5 4.2 4.1  4 3.9 3.8 3.7 ]; % a choice of locations to apply regularization
options.drv=4;  % regularization based on minimization of 4th derivative of F(V) 
options.lam=2e1; % damping parameter magnitude (smoothing by minimization of 4th derivative of F(V) )
options.k=7; % Chosen order of the spline representation

LBF_prf=lin_F_fit(data,options); % do the fit

%
%%%%%%%%%%  Agnostic fits
clear options_ag
options_ag.strainflg='log';
options_ag.Vflg=1;
np=15;
options_ag.knt=linspace(12.73,3.7,np); % distribute knots specified in volume units over the span of the representation 
options_ag.kntflg=1;% ignore previous line and use knots in strain units
options_ag.knt=[  -0.01 0.02  0.04 0.06 0.08 0.10 0.12 0.15 0.17 0.20 0.24 0.27 0.31 0.36 0.42]; % the chosen strains for knots
options_ag.Reg=[linspace(12.7,11,np) 10 9 6  5.5 5 4.8 4.5 4 3.7];
options_ag.drv=4;
options_ag.lam=5e3; % damping set "high"
options_ag.k=6;
LBF_ag=lin_F_fit(data,options_ag); % one agnostic fit

% alternative with less damping
options_ag.lam=6e1; % damping set "low"
LBF_ag2=lin_F_fit(data,options_ag); % the second agnostic fit

%%
%%%%%%%%%% Plot the fits and compare it with previous EoS

load Literature_fits.mat % PV EoS from Myint et al. (2017), Bezacier et al. (2014), Frank et al. (2004), Klotz et al. (2017)

 outd=fn_F_val(LBF_prf,data.PV(end:-1:1,2).^-1);
 pc=outd.P;
 outd=fn_F_val(LBF_ag,data.PV(end:-1:1,2).^-1);
 pc_ag=outd.P;
 Vc=linspace(options.knt(1),options.knt(end)*1.001,1000);
 Vc=linspace(12.73,3.7*1.001,1000);
 rc=Vc.^-1;

% preferred fit
 out=fn_F_val(LBF_prf,rc);
 Pc=out.P;
 Kc=out.K;
 Kp=out.Kp;
 F=out.F;
 
 out2=fn_F_val(LBF_prf2,rc);
 Pc2=out2.P;
 Kc2=out2.K;
 Kp2=out2.Kp;
 F2=out2.F;

% Reference Global fit
out_ref=fn_F_val(LBF_ref,rc); 
Pc_ref=out_ref.P;
Kc_ref=out_ref.K;
Kp_ref=out_ref.Kp;
F_ref=out_ref.F;

% Agnostic fits
out_ag=fn_F_val(LBF_ag,rc); 
Pc_ag=out_ag.P;
Kc_ag=out_ag.K;
Kp_ag=out_ag.Kp;
F_ag=out_ag.F;

out_ag2=fn_F_val(LBF_ag2,rc); 
Pc_ag2=out_ag2.P;
Kc_ag2=out_ag2.K;
Kp_ag2=out_ag2.Kp;
F_ag2=out_ag2.F;


figure1=figure(1);
%%%%%%%%%%%%%%%%% All Data
subplot(4,2,[1 3 5])
shadedplot(P_tot,V_Myint,V_Bez,[1 0.9 0.9], [0/255 158/255 115/255],[213/255 94/255 0/255]);
hold on
plot(P_tot,V_Frank,'-','Color',[0/255 114/255 178/255],'LineWidth',2)
plot(P_tot,V_Klotz,'-','Color',[136/255 34/255 85/255],'LineWidth',2)
plot(data.PV(:,1),data.PV(:,2),'ko','MarkerSize',6,'MarkerFaceColor','k')
plot(Pc_ref,rc.^-1,'-','LineWidth',2,'Color',[.5 .5 .5])
plot(Pc,rc.^-1,'k-','LineWidth',2)
plot(Pc_ag,rc.^-1,'k-.','LineWidth',1.5)
plot(Pc_ag2,rc.^-1,'k:','LineWidth',2)

id=find(Data.VTP.data(:,4)==3);
xlabel('Pressure (GPa)')
ylabel('V (cm^3/mol)')
legend('','Literature fit envelope', 'Myint et al. (2017) <10 GPa','Bezacier et al. (2014) <10 GPa','Frank et al. (2004) <42 GPa','Klotz et al. (2017) <15 GPa','Data')
text(-60,13,'(a)','FontSize',16)
text(-60,2,'(b)','FontSize',16)
text(530,13.3,'(c)','FontSize',16)
text(530,9.8,'(d)','FontSize',16)
text(530,5.7,'(e)','FontSize',16)
text(530,2,'(f)','FontSize',16)

xlim([0 450])
ylim([3 13])
hold off
%%

%%%%%%%%%%%%%%%%% Plot Residual V
subplot(427)
res_Myint = (ppval(pp_V_Myint,Vc')-Pc)./Pc;
res_Bez = (ppval(pp_V_bez,Vc')-Pc)./Pc;
res_Frank = (ppval(pp_V_Frank,Vc')-Pc)./Pc;
res_Klotz = (ppval(pp_V_Klotz,Vc')-Pc)./Pc; 

shadedplot(Pc,res_Bez',res_Myint',[1 0.9 0.9], [1 0.1 0.1],[0.2 0.2 0.4]);
shadedplot(Pc,res_Frank',res_Myint',[1 0.9 0.9], [1 0.2 0.1],[0.2 0.2 0.4]);
shadedplot(Pc,res_Klotz',res_Myint',[1 0.9 0.9], [1 0.1 0.2],[0.2 0.2 0.4]);
plot(pc,(data.PV(end:-1:1,1)-pc)./pc,'ko','MarkerSize',5,'MarkerFaceColor','k')

plot(Pc,(Pc_ag-Pc)./ Pc,'k-.','LineWidth',1.5)
plot(Pc,(ppval(pp_V_bez,Vc')-Pc)./Pc,'-','Color',[213/255 94/255 0/255],'LineWidth',2)
plot(Pc,(ppval(pp_V_Frank,Vc')-Pc)./Pc,'-','Color',[0/255 114/255 178/255],'LineWidth',2)
plot(Pc,(ppval(pp_V_Myint,Vc')-Pc)./Pc,'-','Color',[0/255 158/255 115/255],'LineWidth',2)
plot(Pc,(ppval(pp_V_Klotz,Vc')-Pc)./Pc,'-','Color',[136/255 34/255 85/255],'LineWidth',2)
plot(Pc,(Pc_ref-Pc)./ Pc,'k-','LineWidth',3,'Color',[.5 .5 .5])
plot([2e0 450],[0 0],'k-','LineWidth',1.5)
axis([2e0 450 -0.4 0.4])
xlabel('Pressure (GPa)')
ylabel('\Delta{P}/P fit ')
hold off
set (gca, 'Xscale', 'log');


%%%%%%%%%%%%%%%%% Plot K
subplot(4,2,2)
plot(Data.K.Ahart2011(:,1),Data.K.Ahart2011(:,2),'k*')
hold on
plot(Data.K.Shimizu1995.P(:),Data.K.Shimizu1995.Bs(:),'ks','MarkerSize',10,'MarkerFaceColor','w')
plot(Data.K.Zang2019.All(:,1),Data.K.Zang2019.All(:,5),'kd','MarkerSize',8,'MarkerFaceColor','w')
plot(Data.K.Asahara2010.K_all(:,1),Data.K.Asahara2010.K_all(:,2),'k^','MarkerFaceColor','w')
plot(Pc_ag2,Kc_ag2,'k:','LineWidth',2,'Color','k') 
plot(Pc,Kc,'k-','LineWidth',2)
plot(Pc2,Kc2,'k--','LineWidth',2)
 plot(Pc_ref,Kc_ref,'-','LineWidth',2,'Color',[.5 .5 .5])
   
plot(Pc_ag,Kc_ag,'k-.','LineWidth',1.5)
legend('Ahart (2011)','Shimizu (1995) (P < 8 GPa)', 'Zang (2019)', 'Ashara (2010)','Location','southeast')
 hold off
axis([2e0 450 0 1500])
xlabel('Pressure (GPa)')
ylabel('Bulk Modulus (GPa)')


%%%%%%%%%%%%%%%%% Plot K zoom
subplot(4,2,4)
plot(Pc_ref,Kc_ref,'-','LineWidth',2,'Color',[.5 .5 .5])
hold on
plot(Pc_ag2,Kc_ag2,'k:','LineWidth',2,'Color','k') 
plot(Pc_ag,Kc_ag,'k-.','LineWidth',1.5)
plot(Data.K.Ahart2011(:,1),Data.K.Ahart2011(:,2),'k*')
plot(Data.K.Shimizu1995.P(:),Data.K.Shimizu1995.Bs(:),'ks','MarkerSize',10,'MarkerFaceColor','w')
plot(Data.K.Zang2019.All(:,1),Data.K.Zang2019.All(:,5),'kd','MarkerSize',8,'MarkerFaceColor','w')
plot(Data.K.Asahara2010.K_all(:,1),Data.K.Asahara2010.K_all(:,2),'k^','MarkerFaceColor','w')
plot(Pc,Kc,'k-','LineWidth',2)
plot(Pc2,Kc2,'k--','LineWidth',2)

hold off
axis([0 100 0 500])
xlabel('Pressure (GPa)')
ylabel('Bulk Modulus (GPa)')


%%%%%%%%%%%%%%%%% Plot K'
subplot(4,2,6)
plot([5 5],[-2 10],'-','LineWidth',2,'Color',[.8 .8 .8])
hold on
area([10 10 ; 15 15],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
area([20 20 ; 25 25],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
area([38 38 ; 44 44],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
plot([60 60],[-2 10],'-','LineWidth',2,'Color',[.8 .8 .8])
plot(Pc_ag2,Kp_ag2,'k:','LineWidth',2,'Color','k') 
plot(Pc,Kp,'k-','LineWidth',2)
plot(Pc2,Kp2,'k--','LineWidth',2)
plot(Pc_ref,Kp_ref,'-','LineWidth',2,'Color',[.5 .5 .5])
plot([0 450],[0 0],'k-','LineWidth',0.2)
plot([0 450],[2.5 2.5],'k:')
plot(Pc_ag,Kp_ag,'k-.','LineWidth',2)
hold off
text(-10,500,'(d)','FontSize',16)
 ylim([-2 10])
 xlim([0 450])
 xlabel('Pressure (GPa)')
ylabel('dK/dP ')


%%%%%%%%%%%%%%%%% Plot K' zoom

P_shi = linspace(1,7.5,8);
Kp_shi = -0.30924*P_shi*2 + 8.1239;
subplot(4,2,8)
plot([5 5],[-2 10],'-','LineWidth',3,'Color',[.8 .8 .8])
hold on
plot(P_shi,Kp_shi,'-','LineWidth',4,'Color',[.7 .3 .3])
area([10 10 ; 15 15],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
area([20 20 ; 25 25],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
area([38 38 ; 44 44],[-1.97 11.99 ; -1.97 11.99],'FaceColor',[.8 .8 .8],'LineStyle', 'none')
plot([60 60],[-2 10],'-','LineWidth',5,'Color',[.8 .8 .8])
plot(Pc_ag2,Kp_ag2,'k:','LineWidth',1.5,'Color','k') 
plot(Pc,Kp,'k-','LineWidth',2)
plot(Pc2,Kp2,'k--','LineWidth',2)
plot(Pc_ref,Kp_ref,'-','LineWidth',2,'Color',[.5 .5 .5])
plot(Pc_ag,Kp_ag,'k-.','LineWidth',2) 
plot([0 450],[2.5 2.5],'k:')
plot([0 450],[0 0],'k-')
hold off
 ylim([-2 10])
 xlim([0 100])
 xlabel('Pressure (GPa)')
ylabel('dK/dP ')


